/**
 * A2UI Renderer for TYPO3 Backend
 * Client-side component for handling A2UI interactions
 */
class A2UIRenderer {
  constructor() {
    this.components = new Map();
    this.eventHandlers = new Map();
  }

  /**
   * Render A2UI payload into target element
   */
  async render(payload, targetElement) {
    if (typeof payload === 'string') {
      payload = JSON.parse(payload);
    }

    const container = document.createElement('div');
    container.className = 'a2ui-container';

    for (const componentData of payload.components) {
      const element = await this.renderComponent(componentData);
      if (element) {
        container.appendChild(element);
      }
    }

    // Clear target and append new content
    targetElement.innerHTML = '';
    targetElement.appendChild(container);

    // Initialize event listeners
    this.initializeEventListeners(container);
  }

  /**
   * Render a single component via server
   */
  async renderComponent(componentData) {
    const { id, type, props } = componentData;

    // For demo: create basic HTML structure
    // In production, this would call the TYPO3 backend
    const wrapper = document.createElement('div');
    wrapper.className = 'a2ui-component mb-3';
    wrapper.dataset.componentId = id;
    wrapper.dataset.componentType = type;

    // Store component reference
    this.components.set(id, { type, props, element: wrapper });

    return wrapper;
  }

  /**
   * Initialize event listeners for interactive components
   */
  initializeEventListeners(container) {
    // Handle button clicks
    container.querySelectorAll('button[data-action]').forEach((button) => {
      button.addEventListener('click', (e) => {
        const action = e.target.dataset.action;
        this.handleAction(action, e.target);
      });
    });

    // Handle form submissions
    container.querySelectorAll('form').forEach((form) => {
      form.addEventListener('submit', (e) => {
        e.preventDefault();
        this.handleFormSubmit(form);
      });
    });
  }

  /**
   * Handle component actions
   */
  handleAction(action, element) {
    console.log('A2UI Action:', action, element);

    // Emit custom event for TYPO3 backend
    const event = new CustomEvent('a2ui:action', {
      detail: { action, element },
      bubbles: true,
    });
    element.dispatchEvent(event);
  }

  /**
   * Handle form submission
   */
  handleFormSubmit(form) {
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);

    console.log('A2UI Form Submit:', data);

    // Emit custom event
    const event = new CustomEvent('a2ui:submit', {
      detail: { data, form },
      bubbles: true,
    });
    form.dispatchEvent(event);
  }

  /**
   * Get all collected form data
   */
  getFormData() {
    const data = {};
    this.components.forEach((component, id) => {
      const element = component.element.querySelector('input, select, textarea');
      if (element) {
        if (element.type === 'checkbox') {
          data[id] = element.checked;
        } else {
          data[id] = element.value;
        }
      }
    });
    return data;
  }
}

// Initialize global renderer
window.A2UIRenderer = new A2UIRenderer();

// Auto-initialize on DOM ready
document.addEventListener('DOMContentLoaded', () => {
  // Handle A2UI action events
  document.addEventListener('a2ui:action', (e) => {
    const { action, element } = e.detail;
    
    // Example: Handle specific actions
    switch (action) {
      case 'createPage':
      case 'saveContent':
      case 'updateSeo':
      case 'saveSchedule':
        console.log('Would execute:', action, window.A2UIRenderer.getFormData());
        alert(`Action "${action}" would be executed with the form data.`);
        break;
      default:
        console.log('Unknown action:', action);
    }
  });
});

