# A2UI Integration for TYPO3

**Agent-to-User Interface Protocol Implementation for TYPO3 13/14**

This extension demonstrates the integration of Google's A2UI protocol concepts into TYPO3. It enables AI agents to generate dynamic, context-aware user interfaces in the TYPO3 backend.

## Features

- **Dynamic UI Generation**: AI agent generates forms based on natural language intent
- **Security-First Architecture**: Component whitelist prevents code injection
- **TYPO3 Backend Integration**: Native look and feel with Bootstrap 5 components
- **JSON Payload Support**: Full A2UI protocol compliance (v0.8)

## Requirements

- TYPO3 13.0+ or 14.0+
- PHP 8.2+
- Composer

## Installation

### Via Composer (recommended)

```bash
composer require webconsulting/a2ui-integration
```

### Manual Installation

1. Download the extension
2. Extract to `packages/a2ui_integration/` (TYPO3 13+) or `typo3conf/ext/a2ui_integration/`
3. Run `composer dump-autoload`
4. Activate via Extension Manager or CLI:

```bash
./vendor/bin/typo3 extension:activate a2ui_integration
./vendor/bin/typo3 cache:flush
```

## Usage

1. Navigate to **Web > A2UI Agent Interface** in the TYPO3 backend
2. Enter a natural language request, e.g., "Create a new page" or "Optimize SEO"
3. The agent generates an appropriate form based on your intent

### Example Intents

- "Neue Seite erstellen" → Page creation form
- "Content bearbeiten" → Content editor form
- "SEO optimieren" → SEO metadata form
- "Termin planen" → Scheduling form

## Architecture

```
┌──────────────┐    ┌──────────────┐    ┌──────────────┐
│   Editor     │───►│  Controller  │───►│ AgentService │
└──────────────┘    └──────────────┘    └──────────────┘
                           │                    │
                           ▼                    ▼
                    ┌──────────────┐    ┌──────────────┐
                    │   Renderer   │◄───│  Components  │
                    │   Service    │    │   (A2UI)     │
                    └──────────────┘    └──────────────┘
                           │
                           ▼
                    ┌──────────────┐
                    │  Component   │
                    │  Registry    │
                    │ (Whitelist)  │
                    └──────────────┘
```

## Extending

### Register Custom Components

```php
// In your extension's Services.yaml or ext_localconf.php
$registry = GeneralUtility::makeInstance(ComponentRegistry::class);
$registry->register(
    'my-custom-component',
    'EXT:my_extension/Resources/Private/Templates/Components/MyComponent.html',
    ['label', 'value', 'customProp']
);
```

### Connect to Real LLM

Replace `AgentService::generateUI()` with actual API calls:

```php
public function generateUI(string $intent, array $context = []): array
{
    $response = $this->httpClient->post('https://api.openai.com/v1/chat/completions', [
        'json' => [
            'model' => 'gpt-4',
            'messages' => [
                ['role' => 'system', 'content' => $this->getSystemPrompt()],
                ['role' => 'user', 'content' => $intent],
            ],
        ],
    ]);
    
    $a2uiJson = json_decode($response->getBody(), true);
    return $this->parseComponents($a2uiJson);
}
```

## Security

- **Whitelist-based**: Only registered components can be rendered
- **Prop validation**: Each component type has allowed props defined
- **No code execution**: A2UI sends data, not executable code
- **TYPO3 BE permissions**: Respects TYPO3 backend user access rights

## License

GPL-2.0-or-later

## Author

webconsulting GmbH  
https://webconsulting.at  
office@webconsulting.at

## Resources

- [A2UI Protocol (Google)](https://github.com/google/A2UI)
- [GenUI: AI-Generated Interfaces (Nielsen Norman Group)](https://www.nngroup.com/videos/genui/)
- [TYPO3 Extension Development](https://docs.typo3.org/m/typo3/reference-coreapi/main/en-us/ExtensionArchitecture/Index.html)

