<?php

declare(strict_types=1);

namespace Webconsulting\A2uiIntegration\Service;

use Webconsulting\A2uiIntegration\Domain\Model\A2UIComponent;
use Webconsulting\A2uiIntegration\Domain\Repository\ComponentRegistry;
use TYPO3\CMS\Core\SingletonInterface;
use TYPO3\CMS\Core\Utility\GeneralUtility;
use TYPO3\CMS\Fluid\View\StandaloneView;

/**
 * Renderer Service
 * Renders A2UI components to TYPO3 Backend HTML
 */
class RendererService implements SingletonInterface
{
    public function __construct(
        protected readonly ComponentRegistry $componentRegistry
    ) {}

    /**
     * Render an array of A2UI components
     */
    public function render(array $components): string
    {
        $html = '<div class="a2ui-form">';
        
        foreach ($components as $component) {
            if (!$component instanceof A2UIComponent) {
                continue;
            }
            
            $html .= $this->renderComponent($component);
        }
        
        $html .= '</div>';
        
        return $html;
    }

    /**
     * Render a single A2UI component
     */
    public function renderComponent(A2UIComponent $component): string
    {
        $type = $component->getType();
        
        // Security check: Is this component type registered?
        if (!$this->componentRegistry->isRegistered($type)) {
            return $this->renderError("Unknown component type: {$type}");
        }

        // Security check: Are the props valid?
        if (!$this->componentRegistry->validateProps($type, $component->getProps())) {
            return $this->renderError("Invalid props for component type: {$type}");
        }

        // Get component configuration
        $templatePath = $this->componentRegistry->getTemplatePath($type);
        
        try {
            // Create Fluid view
            $view = GeneralUtility::makeInstance(StandaloneView::class);
            $view->setTemplatePathAndFilename($templatePath);
            
            // Assign component data to view
            $view->assign('id', $component->getId());
            $view->assign('type', $type);
            $view->assign('props', $component->getProps());
            
            // Render children if present
            if ($component->getChildren() !== null) {
                $childrenHtml = '';
                foreach ($component->getChildren() as $child) {
                    $childrenHtml .= $this->renderComponent($child);
                }
                $view->assign('children', $childrenHtml);
            }
            
            return '<div class="a2ui-component mb-3" data-component-id="' 
                . htmlspecialchars($component->getId()) 
                . '" data-component-type="'
                . htmlspecialchars($type)
                . '">'
                . $view->render() 
                . '</div>';
                
        } catch (\Exception $e) {
            return $this->renderError($e->getMessage());
        }
    }

    /**
     * Render an error message
     */
    protected function renderError(string $message): string
    {
        // Only show detailed errors in debug mode
        if ($GLOBALS['TYPO3_CONF_VARS']['BE']['debug'] ?? false) {
            return '<div class="alert alert-danger" role="alert">' 
                . '<strong>A2UI Error:</strong> '
                . htmlspecialchars($message) 
                . '</div>';
        }
        
        return '<!-- A2UI rendering error -->';
    }

    /**
     * Convert components array to A2UI JSON format
     */
    public function toJson(array $components): string
    {
        $data = [
            'version' => '0.8',
            'components' => array_map(
                fn(A2UIComponent $component) => $component->toArray(),
                $components
            ),
        ];
        
        return json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }
}

