<?php

declare(strict_types=1);

namespace Webconsulting\A2uiIntegration\Service;

use Webconsulting\A2uiIntegration\Domain\Model\A2UIComponent;
use TYPO3\CMS\Core\SingletonInterface;

/**
 * Agent Service
 * Simulates an LLM agent that generates A2UI components
 * In production, this would call an actual LLM API (OpenAI, Gemini, etc.)
 */
class AgentService implements SingletonInterface
{
    /**
     * Generate UI components based on context
     * 
     * @param string $intent User intent/query
     * @param array $context Additional context (page type, user role, etc.)
     * @return array Array of A2UIComponent objects
     */
    public function generateUI(string $intent, array $context = []): array
    {
        // This is a simplified demo implementation
        // In production, this would call an LLM API with a proper prompt
        
        return match (true) {
            str_contains(strtolower($intent), 'page') || str_contains(strtolower($intent), 'seite') 
                => $this->generatePageCreationForm($context),
            str_contains(strtolower($intent), 'content') || str_contains(strtolower($intent), 'inhalt') 
                => $this->generateContentEditForm($context),
            str_contains(strtolower($intent), 'schedule') || str_contains(strtolower($intent), 'termin') 
                => $this->generateSchedulingForm($context),
            str_contains(strtolower($intent), 'seo') 
                => $this->generateSEOForm($context),
            default => $this->generateDefaultForm($intent, $context),
        };
    }

    /**
     * Generate a page creation form
     */
    protected function generatePageCreationForm(array $context): array
    {
        return [
            new A2UIComponent(
                'page-title',
                'text-field',
                [
                    'label' => 'Seitentitel',
                    'placeholder' => 'Geben Sie einen aussagekräftigen Titel ein',
                    'required' => true,
                    'maxlength' => 255,
                ]
            ),
            new A2UIComponent(
                'page-type',
                'select',
                [
                    'label' => 'Seitentyp',
                    'options' => [
                        ['value' => '1', 'label' => 'Standard'],
                        ['value' => '3', 'label' => 'Link zu externer URL'],
                        ['value' => '4', 'label' => 'Shortcut'],
                        ['value' => '199', 'label' => 'Spacer'],
                    ],
                    'value' => '1',
                ]
            ),
            new A2UIComponent(
                'page-slug',
                'text-field',
                [
                    'label' => 'URL-Segment (Slug)',
                    'placeholder' => 'automatisch-generiert',
                ]
            ),
            new A2UIComponent(
                'nav-hide',
                'checkbox',
                [
                    'label' => 'Im Menü verstecken',
                    'checked' => false,
                ]
            ),
            new A2UIComponent(
                'publish-date',
                'date-picker',
                [
                    'label' => 'Veröffentlichungsdatum',
                    'value' => date('Y-m-d'),
                    'format' => 'd.m.Y',
                ]
            ),
            new A2UIComponent(
                'submit-btn',
                'button',
                [
                    'label' => 'Seite erstellen',
                    'variant' => 'primary',
                    'action' => 'createPage',
                ]
            ),
        ];
    }

    /**
     * Generate a content editing form
     */
    protected function generateContentEditForm(array $context): array
    {
        return [
            new A2UIComponent(
                'content-header',
                'text-field',
                [
                    'label' => 'Überschrift',
                    'placeholder' => 'H1, H2 oder H3',
                    'required' => true,
                ]
            ),
            new A2UIComponent(
                'content-type',
                'button-group',
                [
                    'options' => ['Text', 'Text & Bild', 'Nur Bild', 'Video', 'HTML'],
                    'selectedValue' => 'Text',
                ]
            ),
            new A2UIComponent(
                'content-body',
                'textarea',
                [
                    'label' => 'Inhalt',
                    'rows' => 10,
                    'placeholder' => 'Geben Sie hier Ihren Content ein...',
                ]
            ),
            new A2UIComponent(
                'save-btn',
                'button',
                [
                    'label' => 'Speichern',
                    'variant' => 'success',
                    'action' => 'saveContent',
                ]
            ),
        ];
    }

    /**
     * Generate a scheduling form
     */
    protected function generateSchedulingForm(array $context): array
    {
        return [
            new A2UIComponent(
                'start-date',
                'date-picker',
                [
                    'label' => 'Veröffentlichung am',
                    'minDate' => date('Y-m-d'),
                    'format' => 'd.m.Y H:i',
                ]
            ),
            new A2UIComponent(
                'end-date',
                'date-picker',
                [
                    'label' => 'Ablaufdatum (optional)',
                    'format' => 'd.m.Y H:i',
                ]
            ),
            new A2UIComponent(
                'schedule-btn',
                'button',
                [
                    'label' => 'Zeitplan festlegen',
                    'variant' => 'primary',
                    'action' => 'saveSchedule',
                ]
            ),
        ];
    }

    /**
     * Generate an SEO optimization form
     */
    protected function generateSEOForm(array $context): array
    {
        return [
            new A2UIComponent(
                'meta-title',
                'text-field',
                [
                    'label' => 'SEO Titel',
                    'placeholder' => 'Max. 60 Zeichen',
                    'maxlength' => 60,
                ]
            ),
            new A2UIComponent(
                'meta-description',
                'textarea',
                [
                    'label' => 'Meta Description',
                    'placeholder' => '150-160 Zeichen empfohlen',
                    'rows' => 3,
                ]
            ),
            new A2UIComponent(
                'focus-keyword',
                'text-field',
                [
                    'label' => 'Focus Keyword',
                    'placeholder' => 'Hauptkeyword für diese Seite',
                ]
            ),
            new A2UIComponent(
                'og-image-btn',
                'button',
                [
                    'label' => 'OG Image wählen',
                    'variant' => 'default',
                    'action' => 'selectOgImage',
                    'icon' => 'actions-image',
                ]
            ),
            new A2UIComponent(
                'update-seo-btn',
                'button',
                [
                    'label' => 'SEO-Daten aktualisieren',
                    'variant' => 'primary',
                    'action' => 'updateSeo',
                ]
            ),
        ];
    }

    /**
     * Generate a default form
     */
    protected function generateDefaultForm(string $intent, array $context): array
    {
        return [
            new A2UIComponent(
                'default-message',
                'text-field',
                [
                    'label' => 'Ihre Anfrage',
                    'placeholder' => 'Beschreiben Sie, was Sie tun möchten...',
                    'value' => $intent,
                    'disabled' => true,
                ]
            ),
            new A2UIComponent(
                'clarification',
                'textarea',
                [
                    'label' => 'Bitte beschreiben Sie genauer, was Sie tun möchten',
                    'rows' => 4,
                    'placeholder' => 'Mehr Details helfen dem Agenten, die richtige UI zu generieren...',
                ]
            ),
            new A2UIComponent(
                'submit-btn',
                'button',
                [
                    'label' => 'Absenden',
                    'variant' => 'primary',
                    'action' => 'processIntent',
                ]
            ),
        ];
    }
}

