<?php

declare(strict_types=1);

namespace Webconsulting\A2uiIntegration\Domain\Repository;

use TYPO3\CMS\Core\SingletonInterface;

/**
 * Component Registry
 * Maintains a whitelist of allowed UI components for security
 */
class ComponentRegistry implements SingletonInterface
{
    /**
     * @var array<string, array>
     */
    protected array $registeredComponents = [];

    public function __construct()
    {
        $this->registerDefaultComponents();
    }

    /**
     * Register a component type
     */
    public function register(string $type, string $templatePath, array $allowedProps = []): self
    {
        $this->registeredComponents[$type] = [
            'templatePath' => $templatePath,
            'allowedProps' => $allowedProps,
        ];
        return $this;
    }

    /**
     * Check if a component type is registered
     */
    public function isRegistered(string $type): bool
    {
        return isset($this->registeredComponents[$type]);
    }

    /**
     * Get component configuration
     */
    public function getComponent(string $type): ?array
    {
        return $this->registeredComponents[$type] ?? null;
    }

    /**
     * Get template path for component
     */
    public function getTemplatePath(string $type): ?string
    {
        return $this->registeredComponents[$type]['templatePath'] ?? null;
    }

    /**
     * Get all registered component types
     */
    public function getRegisteredTypes(): array
    {
        return array_keys($this->registeredComponents);
    }

    /**
     * Validate component props against allowed props
     */
    public function validateProps(string $type, array $props): bool
    {
        if (!$this->isRegistered($type)) {
            return false;
        }

        $config = $this->registeredComponents[$type];
        
        // If no allowed props defined, allow all
        if (empty($config['allowedProps'])) {
            return true;
        }

        // Check if all provided props are in allowed list
        foreach (array_keys($props) as $propKey) {
            if (!in_array($propKey, $config['allowedProps'], true)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Register default TYPO3 Backend components
     */
    protected function registerDefaultComponents(): void
    {
        $basePath = 'EXT:a2ui_integration/Resources/Private/Templates/A2UI/Components/';

        // Text Input
        $this->register('text-field', $basePath . 'TextField.html', [
            'label',
            'placeholder',
            'value',
            'required',
            'maxlength',
            'disabled',
        ]);

        // Textarea
        $this->register('textarea', $basePath . 'Textarea.html', [
            'label',
            'value',
            'rows',
            'placeholder',
            'required',
        ]);

        // Select Dropdown
        $this->register('select', $basePath . 'Select.html', [
            'label',
            'options',
            'value',
            'multiple',
            'required',
        ]);

        // Date Picker
        $this->register('date-picker', $basePath . 'DatePicker.html', [
            'label',
            'value',
            'minDate',
            'maxDate',
            'format',
        ]);

        // Button Group
        $this->register('button-group', $basePath . 'ButtonGroup.html', [
            'options',
            'selectedValue',
            'multiple',
        ]);

        // Button
        $this->register('button', $basePath . 'Button.html', [
            'label',
            'variant',
            'action',
            'icon',
            'disabled',
        ]);

        // Checkbox
        $this->register('checkbox', $basePath . 'Checkbox.html', [
            'label',
            'checked',
            'value',
        ]);
    }
}

