<?php

declare(strict_types=1);

namespace Webconsulting\A2uiIntegration\Domain\Model;

/**
 * A2UI Component Model
 * Represents a single UI component in the A2UI protocol
 */
class A2UIComponent
{
    protected string $id;
    protected string $type;
    protected array $props;
    protected ?array $children = null;

    public function __construct(string $id, string $type, array $props = [], ?array $children = null)
    {
        $this->id = $id;
        $this->type = $type;
        $this->props = $props;
        $this->children = $children;
    }

    public function getId(): string
    {
        return $this->id;
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function getProps(): array
    {
        return $this->props;
    }

    public function getProp(string $key, mixed $default = null): mixed
    {
        return $this->props[$key] ?? $default;
    }

    public function getChildren(): ?array
    {
        return $this->children;
    }

    /**
     * Convert to A2UI JSON format
     */
    public function toArray(): array
    {
        $data = [
            'id' => $this->id,
            'type' => $this->type,
            'props' => $this->props,
        ];

        if ($this->children !== null) {
            $data['children'] = array_map(
                fn(self $child) => $child->toArray(),
                $this->children
            );
        }

        return $data;
    }

    /**
     * Create from A2UI JSON
     */
    public static function fromArray(array $data): self
    {
        $children = null;
        if (isset($data['children']) && is_array($data['children'])) {
            $children = array_map(
                fn(array $childData) => self::fromArray($childData),
                $data['children']
            );
        }

        return new self(
            $data['id'],
            $data['type'],
            $data['props'] ?? [],
            $children
        );
    }
}

