<?php

declare(strict_types=1);

namespace Webconsulting\A2uiIntegration\Controller;

use Psr\Http\Message\ResponseInterface;
use Webconsulting\A2uiIntegration\Service\AgentService;
use Webconsulting\A2uiIntegration\Service\RendererService;
use TYPO3\CMS\Backend\Template\ModuleTemplate;
use TYPO3\CMS\Backend\Template\ModuleTemplateFactory;
use TYPO3\CMS\Extbase\Mvc\Controller\ActionController;

/**
 * A2UI Backend Module Controller
 */
class A2UIController extends ActionController
{
    public function __construct(
        protected readonly ModuleTemplateFactory $moduleTemplateFactory,
        protected readonly AgentService $agentService,
        protected readonly RendererService $rendererService
    ) {}

    /**
     * Dashboard action - Main entry point
     */
    public function dashboardAction(): ResponseInterface
    {
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $moduleTemplate->setTitle('A2UI Agent Interface');
        
        $this->view->assign('examples', [
            'Neue Seite erstellen' => 'create page',
            'Content bearbeiten' => 'edit content',
            'Termin planen' => 'schedule publication',
            'SEO optimieren' => 'seo optimization',
        ]);
        
        $moduleTemplate->setContent($this->view->render());
        return $this->htmlResponse($moduleTemplate->renderContent());
    }

    /**
     * Generate action - Generate UI based on user intent
     */
    public function generateAction(): ResponseInterface
    {
        $intent = '';
        $format = 'html';
        
        if ($this->request->hasArgument('intent')) {
            $intent = (string)$this->request->getArgument('intent');
        }
        
        if ($this->request->hasArgument('format')) {
            $format = (string)$this->request->getArgument('format');
        }

        if (empty($intent)) {
            return $this->jsonResponse(['error' => 'Intent parameter required'], 400);
        }

        // Context could include: current page, user role, backend language, etc.
        $context = [
            'query' => $intent,
            'pageId' => (int)($this->request->getQueryParams()['id'] ?? 0),
            'beUser' => $GLOBALS['BE_USER']->user['username'] ?? 'unknown',
            'language' => $GLOBALS['BE_USER']->uc['lang'] ?? 'default',
        ];

        // Generate UI components via Agent Service
        $components = $this->agentService->generateUI($intent, $context);

        // Return JSON or rendered HTML
        if ($format === 'json') {
            return $this->jsonResponse([
                'success' => true,
                'intent' => $intent,
                'context' => $context,
                'payload' => json_decode($this->rendererService->toJson($components), true),
            ]);
        }

        // Render to HTML
        $html = $this->rendererService->render($components);
        
        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $moduleTemplate->setTitle('Generated UI: ' . htmlspecialchars($intent));
        
        $this->view->assign('intent', $intent);
        $this->view->assign('generatedHtml', $html);
        $this->view->assign('jsonPayload', $this->rendererService->toJson($components));
        
        $moduleTemplate->setContent($this->view->render());
        return $this->htmlResponse($moduleTemplate->renderContent());
    }

    /**
     * Demo action - Show various component examples
     */
    public function demoAction(): ResponseInterface
    {
        $examples = [
            'Seiten-Formular' => $this->agentService->generateUI('create page'),
            'Content-Editor' => $this->agentService->generateUI('edit content'),
            'SEO-Tools' => $this->agentService->generateUI('seo optimization'),
            'Scheduling' => $this->agentService->generateUI('schedule publication'),
        ];

        $renderedExamples = [];
        foreach ($examples as $title => $components) {
            $renderedExamples[$title] = [
                'html' => $this->rendererService->render($components),
                'json' => $this->rendererService->toJson($components),
            ];
        }

        $moduleTemplate = $this->moduleTemplateFactory->create($this->request);
        $moduleTemplate->setTitle('A2UI Component Demo');
        
        $this->view->assign('examples', $renderedExamples);
        
        $moduleTemplate->setContent($this->view->render());
        return $this->htmlResponse($moduleTemplate->renderContent());
    }
}

